unit IWCompTimeEdit;

interface
uses
  {$IFDEF VSNET}
  IWNetComponent,
  System.ComponentModel, System.Drawing,
  IWNetClasses,
  {$ELSE}
  {$IFDEF Linux}IWCLXClasses,{$ELSE}IWVCLClasses,{$ENDIF}
  {$ENDIF}
  Classes,
  IWBaseInterfaces, IWCompEdit, IWRenderContext, IWHTMLTag;

type
  TIWTimeDisplayFormat = (dfAuto, dfMinutes, dfHours, dfDays, dfWeeks);

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWTimeEdit.bmp}
  TIWTimeEdit = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWTimeEdit), 'TIWTimeEdit.bmp')]
  {$ENDIF}
  TIWTimeEdit = class(TIWCustomEdit)
  private
    FMinutes : integer;
    FHoursPerDay : integer;
    FDaysPerWeek : integer;

    FDisplayFormat: TIWTimeDisplayFormat;

    function GetWeeks: string;
    function GetDays: string;
    function GetHours: string;
    procedure SetMinutes(const Value: integer);
    function GetFormattedText : string;
  protected
    procedure InitControl; override;
    procedure SetValue(const AValue: string); override;
  public
    function RenderHTML(AContext: TIWBaseHTMLComponentContext): TIWHTMLTag; override;

    // Not published, they are read only
    property Hours : string read GetHours;
    property Days : string read GetDays;
    property Weeks : string read GetWeeks;
    property Value : string read GetFormattedText;
  published
    property DaysPerWeek : integer read FDaysPerWeek write FDaysPerWeek;
    property HoursPerDay : integer read FHoursPerDay write FHoursPerDay;
    property Minutes : integer read FMinutes write SetMinutes;
    property DisplayFormat : TIWTimeDisplayFormat read FDisplayFormat write FDisplayFormat;
  end;


implementation
uses
  IWResourceStrings,
  SWStrings,SWSystem, SysUtils, IWBaseHTMLControl, IWControl;

{ TIWTimeEdit }

function TIWTimeEdit.GetWeeks: string;
begin
  Result := Format('%2fw', [FMinutes/(60 * FHoursPerDay * FDaysPerWeek)]);
end;

function TIWTimeEdit.GetDays: string;
begin
  Result := Format('%2fd', [FMinutes / (60 * FHoursPerDay)]);
end;

function TIWTimeEdit.GetFormattedText: string;
begin
  if Minutes = 0 then
  begin
    Result := '0';
  end
  else
  begin
    case FDisplayFormat of
      dfAuto    : begin
                    if Minutes < 60 * FHoursPerDay * FDaysPerWeek then
                    begin
                      if Minutes < 60 * FHoursPerDay then
                      begin
                        if Minutes < 60 then
                        begin
                          Result := Format('%dm', [FMinutes])
                        end
                        else
                        begin
                          Result := GetHours;
                        end;
                      end
                      else
                      begin
                        Result := GetDays;
                      end
                    end
                    else
                    begin
                      Result := GetWeeks;
                    end;
                  end;
      dfMinutes : begin
                    Result := Format('%dm', [FMinutes]);
                  end;
      dfHours   : begin
                    Result := GetHours;
                  end;
      dfDays    : begin
                    Result := GetDays;
                  end;
      dfWeeks   : begin
                    Result := GetWeeks;
                  end;
    end;
  end;
end;

function TIWTimeEdit.GetHours: string;
var
  LHours, LMinutes : Integer;
begin
  LHours := FMinutes div 60;
  LMinutes := FMinutes mod 60;
  if LMinutes > 9 then begin
    Result := Format('%d:%d', [LHours,LMinutes]);
  end else begin
    Result := Format('%d:0%d', [LHours,LMinutes]);
  end;
end;

procedure TIWTimeEdit.InitControl;
begin
  inherited InitControl;
  FMinutes := 0;
  FHoursPerDay := 8;
  FDaysPerWeek := 5;
  FDisplayFormat := dfAuto;
end;

function TIWTimeEdit.RenderHTML(
  AContext: TIWBaseHTMLComponentContext): TIWHTMLTag;
begin
  if Visible then
  begin
    (AContext as TIWComponent40Context).AddValidation('!ValidateTimeEdit("' + HTMLName + '")',
      TextToJSStringLiteral(Format(RSInvalidTimeEditFormat, [FriendlyName])));
  end;

  Result := inherited RenderHTML(AContext);
end;

procedure TIWTimeEdit.SetMinutes(const Value: integer);
begin
  FMinutes := Value;
  Text := GetFormattedText;
end;

procedure TIWTimeEdit.SetValue(const AValue: string);
var
  LPart1 : integer;
  LPart2 : double;
  LValue : string;

  LMinutes,
  LHours,
  LDays,
  LWeeks : boolean;
begin
  inherited SetValue(AValue);

  // Set value
  LValue := AValue;
  try
    // Check specifier
    LWeeks := LValue[Length(LValue)] in ['W', 'w'];
    LDays := LValue[Length(LValue)] in ['D', 'd'];
    LHours := (LValue[Length(LValue)] in ['H', 'h']) or (Pos(':', LValue) > 0);
    LMinutes := (LValue[Length(LValue)] in ['M', 'm']) or not(LWeeks or LDays or LHours);

    if LValue[Length(LValue)] in ['D', 'd', 'M', 'm', 'h', 'H', 'W', 'w'] then
    begin
      Delete(LValue, Length(LValue), 1);
    end;

    // Check numeric values
    if Pos(':', LValue) > 0 then  begin
      LPart1 := StrToInt(Fetch(LValue, ':'));
      LPart2 := StrToFloat(LValue);
    end else begin
      if( LHours ) then begin
        LPart1 := StrToInt(LValue);
        LPart2 := 0;
      end else begin
        LPart1 := 0;
        LPart2 := StrToFloat(LValue);
      end;
    end;

    // Compute minutes
    if LWeeks then
    begin
      FMinutes := Trunc(LPart2 * FDaysPerWeek * FHoursPerDay * 60);
    end;

    if LDays then
    begin
      FMinutes := Trunc(LPart2 * FHoursPerDay * 60);
    end;

    if LHours then
    begin
      if Pos('.', AValue) > 0 then
      begin
        FMinutes := Trunc(LPart2 * 60);
      end
      else
      begin
        FMinutes := Trunc(LPart1 * 60 + LPart2);
      end;
    end;

    if LMinutes then
    begin
      FMinutes := Trunc(LPart2);
    end;
  except

    // Too many numbers from the client or other error
    FMinutes := 0;
  end;
end;


end.
